#!/usr/bin/env python

#
# This work is licensed under the terms of the MIT license.
# For a copy, see <https://opensource.org/licenses/MIT>.

"""
Collection of traffic scenarios where the ego vehicle (hero)
is making a left turn
"""
import numpy as np
from six.moves.queue import Queue  # pylint: disable=relative-import

import py_trees
import carla
from agents.navigation.local_planner import RoadOption

from srunner.scenariomanager.carla_data_provider import CarlaDataProvider
from srunner.scenariomanager.scenarioatomics.atomic_behaviors import (ActorTransformSetter,
                                                                      ActorDestroy,
                                                                      ActorSource,
                                                                      ActorSink,
                                                                      WaypointFollower,
                                                                      TrafficLightManipulator)
from srunner.scenariomanager.scenarioatomics.atomic_criteria import CollisionTest
from srunner.scenariomanager.scenarioatomics.atomic_trigger_conditions import InTriggerRegion, DriveDistance
from srunner.scenarios.basic_scenario import BasicScenario
from srunner.tools.scenario_helper import generate_target_waypoint_list

from envs.scenarios.multiagent_basic_scenario import MultiAgentBasicScenario
from envs.utils.parse_scenario_config import convert_dict_to_transform

class UnprotectedLeftTurn(MultiAgentBasicScenario):

    """
    Implementation class for Hero
    Vehicle turning left at signalized junction scenario,
    Traffic Scenario 08.

    This is a single ego vehicle scenario
    """


    def __init__(self, world, ego_vehicles, config, randomize=False, debug_mode=False, criteria_enable=True, timeout=30):
        """
        Setup all relevant parameters and create scenario
        """
        self._world = world
        self._map = CarlaDataProvider.get_map()
        self._brake_value = 0.5
        self._ego_distance = 40
        self._first_vehicle_transform = None
        self._blackboard_queue_name = 'UnprotectedLeftTurn/actor_flow_queue'
        self._queue = py_trees.blackboard.Blackboard().set(self._blackboard_queue_name, Queue())
        self._source_gap = 20
        self._collider_accel_dist = 1 # was 15
        self._target_vel = np.random.uniform(low=32, high=35) / 3.6 # in m/s # 20km/h (32-40)

        self._traffic_light = CarlaDataProvider.get_next_traffic_light(ego_vehicles[0], False)
        # "Topology" of the intersection
        self._annotations = CarlaDataProvider.annotate_trafficlight_in_group(self._traffic_light)
        # TODO: use atomic behavior instead later for multiple scenarios along one route.
        #  For now, assuming only one scenario per route, and starting position is right before trigger location
        RED = carla.TrafficLightState.Red
        YELLOW = carla.TrafficLightState.Yellow
        GREEN = carla.TrafficLightState.Green
        CarlaDataProvider.update_light_states(
            self._traffic_light,
            self._annotations,
            {'ego': GREEN, 'ref': GREEN, 'left': RED, 'right': RED, 'opposite': GREEN},
            freeze=True)

        self._occluder = None
        self._collider = None

        self.timeout = 30.0

        super(UnprotectedLeftTurn, self).__init__("UnprotectedLeftTurn",
                                                         ego_vehicles,
                                                         config,
                                                         world,
                                                         debug_mode,
                                                         criteria_enable=criteria_enable)

    def _initialize_actors(self, config):
        """
        setup the other actor (collider) in the right next lane or the transform in the config
        setup a blocking view truck in the config transform lane at intersection
        """
        other_actor_transform = convert_dict_to_transform(config.other_actors[0].start)
        other_actor_waypoint = CarlaDataProvider.get_map().get_waypoint(
                other_actor_transform.location).previous(5)[0] # Search 10 meters ahead of the collider

        self._first_vehicle_transform = other_actor_waypoint.transform
        # set up collider
        if config.accident_prone:
            first_vehicle = CarlaDataProvider.request_new_actor('vehicle.audi.tt', self._first_vehicle_transform, rolename='collider')
            # first_vehicle.set_simulate_physics(enabled=False)
            self._collider = first_vehicle
            self.other_actors.append(first_vehicle)

        # set up blocking view truck
        other_actor_waypoint_intersection = other_actor_waypoint.next_until_lane_end(1)[-1]
        other_actor_waypoint_intersection = other_actor_waypoint_intersection.next(4)[0]

        self._blocking_truck_transform = carla.Transform(
            carla.Location(other_actor_waypoint_intersection.transform.location.x,
                           other_actor_waypoint_intersection.transform.location.y,
                           1),
            carla.Rotation(other_actor_waypoint_intersection.transform.rotation.pitch,
                           other_actor_waypoint_intersection.transform.rotation.yaw,
                           other_actor_waypoint_intersection.transform.rotation.roll)
            )
        self._occluder = CarlaDataProvider.request_new_actor('vehicle.carlamotors.carlacola',
                                                             self._blocking_truck_transform)
        self._occluder.set_simulate_physics(True)
        self.other_actors.append(self._occluder)
        next_occluder_waypoint = other_actor_waypoint_intersection.previous(6)[0]
        next_occluder = CarlaDataProvider.request_new_actor('vehicle.carlamotors.carlacola',
                                                            next_occluder_waypoint.transform)
        next_occluder.set_simulate_physics(True)
        self.other_actors.append(next_occluder)

        next_occluder_waypoint = next_occluder_waypoint.previous(6)[0]
        next_occluder = CarlaDataProvider.request_new_actor('vehicle.carlamotors.carlacola',
                                                            next_occluder_waypoint.transform)
        next_occluder.set_simulate_physics(True)
        self.other_actors.append(next_occluder)

    def _create_behavior(self):
        """
        Hero vehicle is turning left in an urban area,
        at a signalized intersection, while other actor coming straight
        .The hero actor may turn left either before other actor
        passes intersection or later, without any collision.
        After 80 seconds, a timeout stops the scenario.
        """
        sequence = py_trees.composites.Sequence("UnprotectedLeftTurn")

        actor_source = ActorSource(
            ['vehicle.tesla.model3', 'vehicle.audi.tt'],
            self._first_vehicle_transform, self._source_gap, self._blackboard_queue_name)

        # Generate plan for WaypointFollower
        turn = 0  # LEFT -1, RIGHT 1, STRAIGHT 0
        plan = []

        # generating waypoints until intersection (target_waypoint)
        _, target_waypoint = generate_target_waypoint_list(
            self._map.get_waypoint(self._first_vehicle_transform.location), turn)

        # Generating waypoint list till next intersection
        plan.append((target_waypoint, RoadOption.LANEFOLLOW))
        wp_choice = target_waypoint.next(5.0)
        while len(wp_choice) == 1:
            target_waypoint = wp_choice[0]
            plan.append((target_waypoint, RoadOption.LANEFOLLOW))
            wp_choice = target_waypoint.next(5.0)

        wait = DriveDistance(self.ego_vehicles[0], self._ego_distance)

        # Behavior tree
        root = py_trees.composites.Parallel(
            policy=py_trees.common.ParallelPolicy.SUCCESS_ON_ONE)
        root.add_child(wait)
        root.add_child(actor_source)
        if self._collider is not None:
            move_actor = WaypointFollower(
                self._collider, self._target_vel, plan=plan,
                blackboard_queue_name=self._blackboard_queue_name, avoid_collision=False)
            root.add_child(move_actor)
            sequence.add_child(ActorTransformSetter(self._collider, self._first_vehicle_transform))

        sequence.add_child(root)

        return sequence

    def _create_test_criteria(self):
        """
        A list of all test criteria will be created that is later used
        in parallel behavior tree.
        """
        criteria = []

        # Add approriate checks for other vehicles
        for vehicle in self.other_actors:
           collision_criterion = CollisionTest(vehicle, name="CollisionTest", terminate_on_failure=True)
           criteria.append(collision_criterion)

        return criteria

    def __del__(self):
        self._traffic_light = None
        self.remove_all_actors()
